defmodule Pile do
  @moduledoc """
  Module définissant une pile de cartes.

  On peut ajouter des cartes sur cette pile. Utiliser la fonction
  `ajouter` pour ce faire : la carte ajoutée sera placée au sommet
  de la pile.
  """

  defstruct cartes: []

  @typedoc "Une pile de cartes dans un ordre précis."
  @type t() :: %Pile{cartes: [Carte.t()]}

  @doc """
  Crée une pile vide.

  ## Exemples

      iex> Pile.new
      %Pile{cartes: []}

  """
  @spec new() :: t()
  def new(), do: new(0)

  @doc """
  Crée une pile avec plusieurs cartes déjà présentes.

  Deux valeurs sont possibles : 0 et 52 (0 : pile vide,
  52 : pile avec un jeu complet).

  ## Exemples

      iex> Pile.new(0)
      %Pile{cartes: []}

      iex> pile = Pile.new(52)
      iex> {"1", "pique"} in pile
      true
      iex> {"2", "pique"} in pile
      true
      iex> {"3", "cœur"} in pile
      true
      iex> {"7", "carreau"} in pile
      true
      iex> {"9", "trèfle"} in pile
      true
      iex> Pile.taille(pile)
      52

  """
  @spec new(0 | 52) :: t()
  def new(0), do: %Pile{}

  def new(52) do
    %Pile{cartes: Carte.toutes()}
  end

  @doc """
  Ajoute une carte à la pile.

  Pour ajouter une carte, préciser la pile, ainsi que la valeur
  et l'enseigne de la carte à ajouter. Si la carte a été ajoutée,
  retourne la pile contenant cette nouvelle carte. Sinon, retourne
  l'erreur sous forme d'atome.

  ## Exemples

      iex> Pile.new |> Pile.ajouter("5", "trèfle")
      %Pile{cartes: [%Carte{enseigne: "trèfle", valeur: "5"}]}
      iex> Pile.new |> Pile.ajouter("sept", "trèfle")
      :valeur_invalide
      iex> Pile.new |> Pile.ajouter("7", "autre")
      :enseigne_invalide

  """
  @spec ajouter(t(), String.t(), String.t()) :: t() | Carte.carte_invalide()
  def ajouter(pile, valeur, enseigne) do
    carte = Carte.new(valeur, enseigne)

    case carte do
      %Carte{} ->
        %Pile{cartes: [carte | pile.cartes]}

      erreur ->
        erreur
    end
  end

  @doc """
  Retourne la taille de la pile.

  ## Exemples

      iex> Pile.new |> Pile.taille
      0
      iex> Pile.new |> Pile.ajouter("7", "trèfle") |> Pile.taille
      1

  """
  @spec taille(t()) :: integer()
  def taille(pile), do: length(pile.cartes)

  @doc """
  Mélange semi-aléatoirement les cartes.

  Cette fonction utilise `Enum.shuffle` pour mélanger semi-aléatoirement
  les cartes de la pile. Elle prend en paramètre la pile à mélanger.

  ## Exemples

      iex> # Change la seed aléatoire.
      iex> :rand.seed(:exsss, {100, 101, 102})
      iex> pile = Pile.new(52) |> Pile.mélanger
      iex> Enum.at(pile, 0)
      %Carte{enseigne: "cœur", valeur: "10"}
      iex> Enum.at(pile, 1)
      %Carte{enseigne: "pique", valeur: "6"}
      iex> Pile.taille(pile)
      52

  """
  @spec mélanger(t()) :: t()
  def mélanger(pile) do
    %{pile | cartes: Enum.shuffle(pile)}
  end

  @doc """
  Retire une ou plusieurs cartes de la pile.

  Cette fonction prend en paramètre le nombre de cartes à retirer
  (1 par défaut). Les cartes sont retirées du haut de la pile.
  Elle retourne la pile créée alourdie des cartes retirées
  ainsi que la pile restante.
  Si on cherche à retirer plus de cartes qu'il n'y en a dans la pile,
  la liste des cartes retirées ne contiendra pas le nombre attend.

  ## Exemples

      iex> {retirées, restantes} =
      ...>   Pile.new
      ...>   |> Pile.ajouter("roi", "pique")
      ...>   |> Pile.retirer
      iex> {"roi", "pique"} in retirées
      true
      iex> Pile.taille(retirées)
      1
      iex> Pile.taille(restantes)
      0

      iex> {retirées, restantes} =
      ...>   Pile.new
      ...>   |> Pile.ajouter("9", "cœur")
      ...>   |> Pile.ajouter("5", "trèfle")
      ...>   |> Pile.retirer(2)
      iex> {"9", "cœur"} in retirées
      true
      iex> {"5", "trèfle"} in retirées
      true
      iex> Pile.taille(retirées)
      2
      iex> Pile.taille(restantes)
      0

      iex> {retirées, restantes} =
      ...>   Pile.new
      ...>   |> Pile.ajouter("7", "carreau")
      ...>   |> Pile.retirer(2)
      iex> {"7", "carreau"} in retirées
      true
      iex> Pile.taille(retirées)
      1
      iex> Pile.taille(restantes)
      0

  """
  @spec retirer(t(), number()) :: {t(), t()}
  def retirer(pile, nombre \\ 1) when is_number(nombre) and nombre > 0 do
    retirées = Enum.take(pile, nombre)
    restantes = Enum.drop(pile, nombre)
    {%Pile{cartes: retirées}, %{pile | cartes: restantes}}
  end

  @doc """
  Transfère une carte d'une pile au sommet d'une autre.

  Cette fonction cherche la carte dans la pile d'origine
  (premier argument). Si la carte est trouvée, elle est placée
  au sommet de la seconde pile et retirée de la première. Les deux
  piles (origine et destination) sont retournées dans tous les cas.

  ## Exemples

      iex> origine = Pile.new
      ...> |> Pile.ajouter("dame", "carreau")
      ...> |> Pile.ajouter("5", "trèfle")
      iex> destination = Pile.new |> Pile.ajouter("valet", "trèfle")
      iex> {origine, destination} = Pile.transférer(origine, destination, {"5", "trèfle"})
      iex> {"dame", "carreau"} in origine
      true
      iex> {"5", "trèfle"} in origine
      false
      iex> Pile.taille(origine)
      1
      iex> {"5", "trèfle"} in destination
      true
      iex> {"valet", "trèfle"} in destination
      true
      iex> Pile.taille(destination)
      2

      iex> origine = Pile.new
      ...> |> Pile.ajouter("dame", "pique")
      ...> |> Pile.ajouter("5", "carreau")
      iex> destination = Pile.new |> Pile.ajouter("valet", "trèfle")
      iex> {origine, destination} = Pile.transférer(origine, destination, Carte.new("5", "carreau"))
      iex> {"dame", "pique"} in origine
      true
      iex> {"5", "carreau"} in origine
      false
      iex> Pile.taille(origine)
      1
      iex> {"5", "carreau"} in destination
      true
      iex> {"valet", "trèfle"} in destination
      true
      iex> Pile.taille(destination)
      2

  """
  @spec transférer(t(), t(), Carte.t() | {String.t(), String.t()}) :: {t(), t()}
  def transférer(origine, destination, {valeur, enseigne}) do
    transférer(origine, destination, Carte.new(valeur, enseigne))
  end

  def transférer(origine, destination, carte) do
    if carte in origine do
      origine = %{origine | cartes: List.delete(origine.cartes, carte)}
      destination = %{destination | cartes: [carte | destination.cartes]}
      {origine, destination}
    else
      {origine, destination}
    end
  end

  @doc """
  Fusionne deux piles de cartes.

  Cette fonction prend deux piles de cartes en paramètre et retourne
  la pile les combinant.

  ## Exemples

      iex> première = Pile.new |> Pile.ajouter("3", "trèfle")
      iex> seconde = Pile.new |> Pile.ajouter("valet", "trèfle")
      iex> pile = Pile.fusionner(première, seconde)
      iex> {"3", "trèfle"} in pile
      true
      iex> {"valet", "trèfle"} in pile
      true

  """
  def fusionner(première, seconde) do
    %Pile{cartes: Enum.concat(première, seconde)}
  end

  defimpl Enumerable do
    def count(pile), do: {:ok, Pile.taille(pile)}

    def member?(pile, {valeur, enseigne}) do
      carte = Carte.new(valeur, enseigne)
      {:ok, carte in pile.cartes}
    end

    def member?(pile, carte) do
      {:ok, carte in pile.cartes}
    end

    def reduce(pile, acc, fun) do
      Enumerable.List.reduce(pile.cartes, acc, fun)
    end

    def slice(%Pile{cartes: []}), do: {:ok, 0, fn _, _, _ -> [] end}
    def slice(_list), do: {:error, __MODULE__}
  end

  defimpl Inspect do
    def inspect(pile, options) do
      cartes = Enum.map(pile.cartes, fn carte -> Carte.nom(carte) end)
      Inspect.Algebra.concat(["#Pile", Inspect.Algebra.to_doc(cartes, options)])
    end
  end
end
